<?php
/* 
 * Add-on Name:	WooCommerce Invoice PDF
 * Description:	This plugin adds an Invoice PDF as an attachment to customer emails, enables backend download of the pdf and customer download on the my account page
 * Version:		1.0.4.1
 * Author:		Stil-Etage GmbH
 * Author URI:	http://stil-etage.de
 * Licence:		GPLv3
 * Text Domain:	woocommerce-invoice-pdf
 * Last Change: 2016-04-07
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
} 

use Dompdf\Dompdf;

if ( ! class_exists( 'Woocommerce_Invoice_Pdf' ) ) {

	/**
	* main class for plugin
	*
	* @class Woocommerce_Invoice_Pdf
	* @version 1.0
	* @category	Class
	*/ 
	class Woocommerce_Invoice_Pdf {
		
		/**
		* singleton
		* @var object
		*/
		static $instance = NULL;
		
		 /**
		 _e(* @var string
		 */
		static public $plugin_filename = __FILE__;
		
		/**
		* singleton getInstance
		*
		* @since 0.0.1
		* @access public
		* @static
		* @hook plugins_loaded
		* @return class Woocommerce_Invoice_Pdf
		*/			
		public static function get_instance() {
			if ( self::$instance == NULL) {
				self::$instance = new Woocommerce_Invoice_Pdf();	
			}
			return self::$instance;
		}
		
		/**
		* constructor
		*
		* @since 0.0.1
		* @access private
		* @return void
		*/	
		private function __construct() {
			
			// if mb_string is missing
			require_once( untrailingslashit( plugin_dir_path( __FILE__ ) ) . DIRECTORY_SEPARATOR . 'vendors' . DIRECTORY_SEPARATOR . 'self' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'mb_string_missing.php' );

			// load dompdf autoloader
			define( 'WP_WC_INVOICE_PDF_DOMPDF_LIB_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) . DIRECTORY_SEPARATOR . 'vendors' . DIRECTORY_SEPARATOR . 'dompdf' );

			// auto-load classes on demand
			if ( function_exists( "__autoload" ) ) {
				spl_autoload_register( "__autoload" );
			}

			spl_autoload_register( array( $this, 'autoload' ) );
			// define cache directory
			if ( ! defined( 'WP_WC_INVOICE_PDF_CACHE_DIR' ) ) {
				define( 'WP_WC_INVOICE_PDF_CACHE_DIR', untrailingslashit( WP_CONTENT_DIR ) . DIRECTORY_SEPARATOR . 'cache' . DIRECTORY_SEPARATOR . 'woocommerce-invoice-pdf' . DIRECTORY_SEPARATOR );
			}
			self::init();

		}
		
		/**
		* autoload classes on demand
		*
		* @since 0.0.1
		* @access public
		* @arguments string $class (class name)
		* @return void
		*/
		public function autoload( $class ) {
			
			$filename 		= false;
			$class			= strtolower( $class );
			$file          	= 'class-' . str_replace( '_', '-', $class ) . '.php';
			$file 			= str_replace( 'class-wp-wc-invoice-pdf-', '', $file );
			$vendors_path	= untrailingslashit( plugin_dir_path( __FILE__ ) ) . DIRECTORY_SEPARATOR . 'vendors';
			
			if ( $class == 'ebs_pdf_wordpress' ) {
				$file = $vendors_path . DIRECTORY_SEPARATOR . 'ebs-pdf' . DIRECTORY_SEPARATOR . $file;
			} else if ( strpos( $class, 'wp_wc_invoice_pdf_backend_' ) === 0 ){
				$applications_backend_path = $vendors_path . DIRECTORY_SEPARATOR . 'self' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'be';	
				$file = $applications_backend_path . DIRECTORY_SEPARATOR . $file;
			} else {
				$applications_path = $vendors_path . DIRECTORY_SEPARATOR . 'self' . DIRECTORY_SEPARATOR . 'app';			
				$file = $applications_path . DIRECTORY_SEPARATOR . $file;
			}
			
			if ( $file && is_readable( $file ) ) {
				include_once( $file );
			}
			
		}
		
		/**
		* cloning is private
		*
		* @since 0.0.1
		*/	
		private function __clone() {}
		
		/**
		* register actions and filters
		*
		* @since 0.0.1
		* @access private
		* @static
		* @return void
		*/		
		private static function init() {	

			// email attachment
			add_filter( 'woocommerce_email_attachments', array( 'WP_WC_Invoice_Pdf_Email_Attachment', 'add_attachment' ), 10, 3 );

			// email attachment for refund emails
			add_action( 'woocommerce_order_fully_refunded_notification',  		array( 'WP_WC_Invoice_Pdf_Email_Attachment', 'refunded_trigger' ), 10, 2 );
			add_action( 'woocommerce_order_partially_refunded_notification',	array( 'WP_WC_Invoice_Pdf_Email_Attachment', 'refunded_trigger' ), 10, 2 );

			// additional pdfs as email attachments
			add_filter( 'woocommerce_email_attachments', array( 'WP_WC_Invoice_Pdf_Email_Attachment', 'additional_email_attachments' ), 10, 3 );
			
			// plugin has been activated
			if ( is_admin() && current_user_can( 'activate_plugins' ) && get_option( 'wp_wc_invoice_pdf_just_activated', false ) ){
				
				// admin_notices
				//add_filter( 'admin_notices', array( 'WP_WC_Invoice_Pdf_Backend_Activation', 'output_notices' ) );
				add_filter( 'admin_init', array( 'WP_WC_Invoice_Pdf_Backend_Activation', 'load_defaults' ) ) ;
				add_filter( 'admin_init', array( 'WP_WC_Invoice_Pdf_Backend_Activation', 'setup_cache_dir' ) ) ;
			}
			
			// option page
			if ( is_admin() ) {
				add_filter( 'woocommerce_de_ui_left_menu_items',								array( 'WP_WC_Invoice_Pdf_Backend_Options_WGM', 'menu' ) );
				add_action( 'woocommerce_admin_field_wp_wc_invoice_pdf_textarea', 				array( 'WP_WC_Invoice_Pdf_Backend_Options_WGM', 'output_textarea' ) );
				add_action( 'woocommerce_admin_field_wp_wc_invoice_pdf_test_download_button', 	array( 'WP_WC_Invoice_Pdf_Backend_Options_WGM', 'output_test_pdf_button' ) );
				add_filter( 'woocommerce_admin_settings_sanitize_option', 						array( 'WP_WC_Invoice_Pdf_Backend_Options_WGM', 'save_wp_wc_invoice_pdf_textarea_textarea'), 10, 3 );
				add_filter( 'woocommerce_admin_settings_sanitize_option',						array( 'WP_WC_Invoice_Pdf_Backend_Options_WGM', 'save'), 10, 3 );
				add_action( 'admin_enqueue_scripts',											array( __CLASS__, 'media_uploader_scripts' ) );	// scripts and styles to use media uploader for image upload
			}
			
			// backend download buttons
			if ( is_admin() ) {
				add_action( 'admin_enqueue_scripts',												array( __CLASS__, 'admin_styles' ) ); // style for download button
				add_filter( 'woocommerce_order_actions',											array( 'WP_WC_Invoice_Pdf_Backend_Download', 'order_action_option' ) );
				add_action( 'woocommerce_order_action_wp_wc_invoice_pdf_invoice',					array( 'WP_WC_Invoice_Pdf_Backend_Download', 'order_action' ) );

				add_filter( 'woocommerce_admin_order_actions',										array( 'WP_WC_Invoice_Pdf_Backend_Download', 'admin_icon_download'), 10, 2 );
				add_action( 'wp_ajax_woocommerce_wp_wc_invoice_pdf_invoice_download', 				array( 'WP_WC_Invoice_Pdf_Backend_Download', 'admin_ajax_download_pdf' ) );
				add_action( 'wp_ajax_woocommerce_wp_wc_invoice_pdf_invoice_delete_content',			array( 'WP_WC_Invoice_Pdf_Backend_Download', 'invoice_pdf_delete_saved_content' ) );
				add_action( 'admin_notices', 														array( 'WP_WC_Invoice_Pdf_Backend_Download', 'admin_notices' ) );
				add_action( 'wp_ajax_woocommerce_wp_wc_invoice_pdf_test_invoice', 					array( 'WP_WC_Invoice_Pdf_Backend_Download', 'admin_ajax_test_invoice' ) );
				add_filter( 'wgm_refunds_actions',													array( 'WP_WC_Invoice_Pdf_Backend_Download', 'admin_refund_icon_download' ), 10, 2 );
				add_action( 'wp_ajax_woocommerce_wp_wc_invoice_pdf_refund_download',				array( 'WP_WC_Invoice_Pdf_Backend_Download', 'admin_ajax_download_refund_pdf' ) );
				add_action( 'wp_ajax_woocommerce_wp_wc_invoice_pdf_refund_delete_saved_content',	array( 'WP_WC_Invoice_Pdf_Backend_Download', 'admin_ajax_refund_delete_saved_content' ) );
				if ( get_option( 'wp_wc_invoice_pdf_new_post_message', false ) == true ) {
					add_filter( 'admin_notices', array( 'WP_WC_Invoice_Pdf_Backend_Download', 'output_notices' ) );
				}

				// bulk zip download since GM 3.1
				add_action( 'admin_footer', 														array( 'WP_WC_Invoice_Pdf_Backend_Download', 'bulk_admin_footer' ), 10 );
				add_action( 'load-edit.php', 														array( 'WP_WC_Invoice_Pdf_Backend_Download', 'bulk_action' ) );
				add_action( 'wgm_refunds_render_refund_id',											array( 'WP_WC_Invoice_Pdf_Backend_Download', 'refund_checkboxes' ), 10, 2 );
				add_action( 'wgm_refunds_render_refund_id_head',									array( 'WP_WC_Invoice_Pdf_Backend_Download', 'refund_checkboxes_select_all' ) );
				add_action( 'woocommerc_de_refund_before_list',										array( 'WP_WC_Invoice_Pdf_Backend_Download', 'submit_button' ) );
				add_action( 'woocommerc_de_refund_after_list',										array( 'WP_WC_Invoice_Pdf_Backend_Download', 'submit_button' ) );
				add_action( 'admin_init',															array( 'WP_WC_Invoice_Pdf_Backend_Download', 'bulk_action_refunds' ) );

			}
			
			// frontend download button on my account page view-order 
			if ( ! is_admin() ) {
				add_action( 'woocommerce_order_details_after_order_table', 							array( 'WP_WC_Invoice_Pdf_View_Order_Download', 'make_download_button' ) );
			}
			
			add_action( 'wp_ajax_woocommerce_wp_wc_invoice_pdf_view_order_invoice_download', 	array( 'WP_WC_Invoice_Pdf_View_Order_Download', 'download_pdf' ) );
	
		}
		
		/**
		* enqueue css file for download button design on shop order page
		*
		* @since 0.0.1
		* @access public
		* @static
		* @hook admin_enqueue_scripts
		* @return void
		*/				
		public static function admin_styles() {
			if ( get_current_screen()->id == 'edit-shop_order' || get_current_screen()->id == 'woocommerce_page_wgm-refunds' ) { // add style only if we need it
				$min = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : 'min.';
				wp_enqueue_style( 'woocommerce_wp_wc_invoice_pdf_admin_styles', untrailingslashit( plugin_dir_url( __FILE__ ) ) . '/vendors/self/assets/css/admin.' . $min . 'css' );
			}
		}
		
		/**
		* plugin activation
		*
		* @since 0.0.1
		* @access public
		* @static
		* @hook register_activation_hook
		* @return void
		*/
		public static function activate(){
			$vendors_path	= untrailingslashit( plugin_dir_path( __FILE__ ) ) . DIRECTORY_SEPARATOR . 'vendors';
			$backend_path	= $vendors_path . DIRECTORY_SEPARATOR . 'self' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'be';
			include_once( $backend_path . DIRECTORY_SEPARATOR . 'backend-activation.php' );
			WP_WC_Invoice_Pdf_Backend_Activation::activation();
		}

		/**
		* plugin deactivation
		*
		* @since 1.0.4.1
		* @access public
		* @static
		* @hook register_deactivation_hook
		* @return void
		*/
		public static function deactivate(){
			// remove cache
			WP_WC_Invoice_Pdf_Create_Pdf::clear_cache();
			if ( is_dir( WP_WC_INVOICE_PDF_CACHE_DIR ) ) {
				rmdir( WP_WC_INVOICE_PDF_CACHE_DIR );
			}
			//WP_WC_Invoice_Pdf_Backend_Download::clear_zip_cache( false, true );
		}
		
		/**
		* enqueue scripts and styles to enable media uploader for image upload on the settings page
		*
		* @since 0.0.1
		* @access public
		* @static
		* @hook admin_enqueue_scripts
		* @return void
		*/	
		public static function media_uploader_scripts() {
			
			$min = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : 'min.';

			if ( ( ( get_current_screen()->id == 'woocommerce_page_german-market' ) && isset( $_GET[ 'tab' ] ) && ( $_GET[ 'tab' ] == 'invoice-pdf' ) && isset( $_GET[ 'sub_tab' ] ) && ( $_GET[ 'sub_tab' ] == 'images' ) ) || get_current_screen()->id == 'woocommerce_page_wgm-refunds' ){
				
				wp_register_script( 'wp-wc-invoice-pdf-media-uploader', untrailingslashit( plugin_dir_url( __FILE__ ) ) . '/vendors/self/assets/js/admin.' . $min . 'js', array( 'jquery' ) );
				wp_enqueue_script( 'wp-wc-invoice-pdf-media-uploader' );
			
			} else if ( get_current_screen()->id == 'edit-shop_order' ) {
				wp_register_script( 'wp-wc-invoice-pdf-media-uploader', untrailingslashit( plugin_dir_url( __FILE__ ) ) . '/vendors/self/assets/js/admin.' . $min . 'js', array( 'jquery' ) );
				wp_enqueue_script( 'wp-wc-invoice-pdf-media-uploader' );
			}

		}

	} // end class

} // end class exists

Woocommerce_Invoice_Pdf::get_instance();
